/*+ CommPort.cpp
 *
 ******************************************************************************
 *
 *                        Trimble Navigation Limited
 *                           645 North Mary Avenue
 *                              P.O. Box 3642
 *                         Sunnyvale, CA 94088-3642
 *
 ******************************************************************************
 *
 *    Copyright  2005 Trimble Navigation Ltd.
 *    All Rights Reserved
 *
 ******************************************************************************
 *
 * Description:
 *    This file implements the CCommPort class.
 *            
 * Revision History:
 *    05-18-2005    Mike Priven
 *                  Written
 *
 * Notes:
 *
-*/

/*---------------------------------------------------------------------------*\
 |                         I N C L U D E   F I L E S
\*---------------------------------------------------------------------------*/
#include "stdafx.h"
#include "CommPort.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


/*---------------------------------------------------------------------------*\
 |                    M E T H O D   D E F I N I T I O N S
\*---------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
Function:       <constructor>

Description:    A constructor for this class. Called when a new object of this
                class is created.

Parameters:     pParent - a pointer to the parent window - the window that
                created an object of this class.

Return Value:   none
-----------------------------------------------------------------------------*/
CCommPort::CCommPort (CWnd* pWnd)
{ 
    OSVERSIONINFO tOsVersionInfo;

    tOsVersionInfo.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
    GetVersionEx(&tOsVersionInfo);

    m_bOverlapped     = (tOsVersionInfo.dwPlatformId == VER_PLATFORM_WIN32_NT);
    m_pParent         = pWnd; 
    m_hPort           = INVALID_HANDLE_VALUE; 
    m_ucPortNum       = 0; 
    m_ulBaud          = COM_BAUD_9600; 
    m_ucDataBits      = COM_DATA_BITS_8; 
    m_ucParity        = COM_PARITY_ODD; 
    m_ucStopBits      = COM_STOP_BITS_1;
    memset(&m_tOvRead, '\0', sizeof(m_tOvRead)); 
    memset(&m_tOvWrite, '\0', sizeof(m_tOvWrite)); 
    m_tOvRead.hEvent  = NULL;
    m_tOvWrite.hEvent = NULL;
}


/******************************************************************************
 *
 *    Function Name:    Open()
 *
 *    Description:
 *        Opens the port with the specified parameters.
 *
 *    Input Parameters:
 *        ucPortNum  - port number to open
 *        ulBaud     - transmission baud rate
 *        ucParity   - parity
 *        ucDataBits - number of data bits per byte
 *        ucStopBits - number of stop bits per byte
 *        
 *    Output Parameters:
 *        None
 *        
 *    Return Value:
 *        true if port successfully open; false if an error occurred.
 *
 *****************************************************************************/
bool CCommPort::Open (U8   ucPortNum, 
                      U32  ulBaud, 
                      U8   ucParity, 
                      U8   ucDataBits, 
                      U8   ucStopBits)
{
    CString      strPort;
    COMMTIMEOUTS tTimeOuts;

    if (m_hPort != INVALID_HANDLE_VALUE)
    {
        Close();
    }

    if (!SetPort (ucPortNum))
    {
        return false;
    }

    /* Create a COM port string from the integer port number */
    if (ucPortNum < 10 || !m_bOverlapped)
    {
        strPort.Format ("COM%d", ucPortNum);
    }
    else if (m_bOverlapped)
    {
        strPort.Format ("\\\\.\\COM%d", ucPortNum);
    }

    m_hPort = CreateFile (strPort,
                          GENERIC_READ|GENERIC_WRITE,
                          0,
                          NULL,
                          OPEN_EXISTING,
                          ((m_bOverlapped) ? FILE_FLAG_OVERLAPPED : 0),
                          NULL);

    if (m_hPort == INVALID_HANDLE_VALUE)
    {
        Close();
        return false;
    }

    if (m_bOverlapped) 
    {
        if (m_tOvRead.hEvent != NULL)
        {
            CloseHandle (m_tOvRead.hEvent);
            m_tOvRead.hEvent = NULL;
        }

        if ((m_tOvRead.hEvent = CreateEvent(NULL, TRUE,  FALSE, NULL)) == NULL) 
        {
            Close();
            return false;
        }

        if (m_tOvWrite.hEvent != NULL)
        {
            CloseHandle (m_tOvWrite.hEvent);
            m_tOvWrite.hEvent = NULL;
        }

        if ((m_tOvWrite.hEvent = CreateEvent(NULL, TRUE, FALSE, NULL)) == NULL) 
        {
            Close();
            return false;
        }
    }

    if (SetupComm (m_hPort, 4096, 4096) == FALSE) 
    {
        Close();
        return false;
    }

    if (!SetPortState (ulBaud, ucParity, ucDataBits, ucStopBits)) 
    {
        Close();
        return false;
    }

    tTimeOuts.ReadIntervalTimeout         = MAXDWORD;
    tTimeOuts.ReadTotalTimeoutMultiplier  = 0;
    tTimeOuts.ReadTotalTimeoutConstant    = 0;
    tTimeOuts.WriteTotalTimeoutMultiplier = 0;
    tTimeOuts.WriteTotalTimeoutConstant   = 0;

    if (SetCommTimeouts (m_hPort, &tTimeOuts) == FALSE) 
    {
        Close();
        return false;
    }

    return true;
}


/******************************************************************************
 *
 *    Function Name:    Open()
 *
 *    Description:
 *        Opens the port with internally-set parameters. These parameters must
 *        be set first with SetXXXXX() functions.
 *
 *    Input Parameters:
 *        None
 *        
 *    Output Parameters:
 *        None
 *        
 *    Return Value:
 *        true if no error; false otherwise
 *
 *****************************************************************************/
bool CCommPort::Open ()
{
    return Open(m_ucPortNum,m_ulBaud,m_ucParity,m_ucDataBits,m_ucStopBits);
}


/******************************************************************************
 *
 *    Function Name:    Close()
 *
 *    Description:
 *        Closes the port.
 *
 *    Input Parameters:
 *        None
 *        
 *    Output Parameters:
 *        None
 *        
 *    Return Value:
 *        true if no error; false otherwise
 *
 *****************************************************************************/
bool CCommPort::Close()
{
    bool bSuccess = true;

    if (CloseHandle (m_hPort) == FALSE)
    {
        bSuccess = false;
    }
    else
    {
        m_hPort = INVALID_HANDLE_VALUE;
    }

    if (m_bOverlapped) 
    {
        if (CloseHandle (m_tOvRead.hEvent) == FALSE)
        {
            bSuccess = false;
        }
        else
        {
            m_tOvRead.hEvent = NULL;
        }

        if (CloseHandle (m_tOvWrite.hEvent) == FALSE)
        {
            bSuccess = false;
        }
        else
        {
            m_tOvWrite.hEvent = NULL;
        }
    }

    return bSuccess;
}


/******************************************************************************
 *
 *    Function Name:    Reset()
 *
 *    Description:
 *        Resets the port with the current parameters
 *
 *    Input Parameters:
 *        None
 *        
 *    Output Parameters:
 *        None
 *        
 *    Return Value:
 *        true if no error; false otherwise
 *
 *****************************************************************************/
bool CCommPort::Reset()
{
    if (m_hPort == INVALID_HANDLE_VALUE)
    {
        return false;
    }

    if (!SetPortState (m_ulBaud, m_ucParity, m_ucDataBits, m_ucStopBits))
    {
        return false;
    }

    return true;
}


/******************************************************************************
 *
 *    Function Name:    SetPortState()
 *
 *    Description:
 *        Sets the port state with the specified parameters.
 *
 *    Input Parameters:
 *        ulBaud     - transmission baud rate
 *        ucParity   - parity
 *        ucDataBits - number of data bits per byte
 *        ucStopBits - number of stop bits per byte
 *        
 *    Output Parameters:
 *        None
 *        
 *    Return Value:
 *        true if no error; false otherwise
 *
 *****************************************************************************/
bool CCommPort::SetPortState (U32 ulBaud, 
                              U8  ucParity, 
                              U8  ucDataBits,
                              U8  ucStopBits)
{
    DCB tDevCntrlBlock;

    if (m_hPort == INVALID_HANDLE_VALUE)
    {
        return false;
    }

    if (!SetBaud     (ulBaud)     || 
        !SetParity   (ucParity)   ||
        !SetStopBits (ucStopBits) ||
        !SetDataBits (ucDataBits))
    {
        return false;
    }

    if (PurgeComm (m_hPort, PURGE_RXCLEAR) == FALSE)
    {
       return false;
    }

    if (GetCommState (m_hPort, &tDevCntrlBlock) == 0)
    {
        return false;
    }

    tDevCntrlBlock.fBinary           = TRUE;
    tDevCntrlBlock.BaudRate          = ulBaud;
    tDevCntrlBlock.Parity            = ucParity;
    tDevCntrlBlock.ByteSize          = ucDataBits;
    tDevCntrlBlock.StopBits          = ucStopBits;
    tDevCntrlBlock.fDtrControl       = DTR_CONTROL_DISABLE;
    tDevCntrlBlock.fRtsControl       = RTS_CONTROL_DISABLE;
    tDevCntrlBlock.fOutxCtsFlow      = 0;
    tDevCntrlBlock.fOutxDsrFlow      = 0;
    tDevCntrlBlock.fDsrSensitivity   = 0;
    tDevCntrlBlock.fTXContinueOnXoff = 0;
    tDevCntrlBlock.fOutX             = 0;
    tDevCntrlBlock.fInX              = 0;
    tDevCntrlBlock.fErrorChar        = 0;
    tDevCntrlBlock.fNull             = 0;
    tDevCntrlBlock.fAbortOnError     = 0;

    if (SetCommState (m_hPort, &tDevCntrlBlock) == FALSE)
    {
        return false;
    }

    if (PurgeComm (m_hPort, PURGE_TXCLEAR | PURGE_RXCLEAR) == FALSE)
    {
        return false;
    }

    return  true;
}


/******************************************************************************
 *
 *    Function Name:    Write()
 *
 *    Description:
 *        Writes a number of byte to the port.
 *
 *    Input Parameters:
 *        pucBuf   - a pointer to the byte buffer to write to the port
 *        ulBufLen - number of bytes to write
 *        
 *    Output Parameters:
 *        None
 *        
 *    Return Value:
 *        Number of bytes successfully written to the port; 
 *        -1 if error occurred
 *
 *****************************************************************************/
S32 CCommPort::Write (U8 *pucBuf, U16 usBufLen)
{
    U32 ulBytesWritten;

    if (m_hPort == INVALID_HANDLE_VALUE)
    {
        return -1;
    }

    if (pucBuf == NULL)
    {
       return -1;
    }

    if (m_bOverlapped) 
    {
        if (!WriteFile (m_hPort, 
                        pucBuf, 
                        usBufLen, 
                        &ulBytesWritten, 
                        &m_tOvWrite))
        {
            if (!GetOverlappedResult (m_hPort,
                                      &m_tOvWrite, 
                                      &ulBytesWritten, 
                                      TRUE))
            {
                return -1;
            }
        }
    }
    else
    {
        WriteFile (m_hPort, 
                   pucBuf, 
                   usBufLen, 
                   &ulBytesWritten, 
                   NULL);
    }

    return (S32)ulBytesWritten;
}


/******************************************************************************
 *
 *    Function Name:    Write()
 *
 *    Description:
 *        Writes a byte to the port.
 *
 *    Input Parameters:
 *        ucData - a byte to write to the port.
 *        
 *    Output Parameters:
 *        None
 *        
 *    Return Value:
 *        1 if the byte successfully written; -1 if an error occurred.
 *
 *****************************************************************************/
S32 CCommPort::Write (U8 ucData)
{
    return Write (&ucData, 1);
}


/******************************************************************************
 *
 *    Function Name:    Read()
 *
 *    Description:
 *        Reads a number of bytes from the port and places them in the buffer.
 *
 *    Input Parameters:
 *        ulBufLen - number of bytes to read
 *        
 *    Output Parameters:
 *        pucBuf   - a pointer to the buffer where the bytes will be placed
 *        
 *    Return Value:
 *        Number of bytes successfully read; -1 if an error occurred.
 *
 *****************************************************************************/
S32 CCommPort::Read (U8 *pucBuf, U16 usBufLen)
{
    U32        ulBytesRead = 0;
    U16        usIdx = 0;
    OVERLAPPED *ptOverlapped;
    U16        usNumToRead;
        
    if (m_hPort == INVALID_HANDLE_VALUE)
    {
        return -1;
    }
    
    if (pucBuf == NULL)
    {
       return -1;
    }
    
    ptOverlapped = (m_bOverlapped) ? &m_tOvRead : NULL;

    switch (usBufLen) 
    {
        case 1:
            while (ulBytesRead != 1) 
            {       
                if ((ReadFile (m_hPort, 
                               &pucBuf[usIdx], 
                               1, 
                               &ulBytesRead, 
                               ptOverlapped) == 0) || !ulBytesRead)
                {
                    Sleep(1);
                }
            } 
            usIdx = 1;              
            break;

        default:
            usNumToRead = usBufLen;
            usIdx = 0;
            while (usIdx != usBufLen)
            {
                if (ReadFile (m_hPort, 
                              &pucBuf[usIdx],
                              usNumToRead, 
                              &ulBytesRead, 
                              ptOverlapped)) 
                {
                    if (ulBytesRead == usBufLen) 
                    {
                        usIdx = usBufLen;
                    } 
                    else if (usBufLen > 0 && ulBytesRead)
                    {
                        usIdx += (U16)ulBytesRead;
                        usNumToRead = usBufLen - usIdx;
                    }
                    else 
                    {
                        Sleep (1);
                    }
                }
                else
                {
                    Sleep (1);
                }
            }
            break;
    }

    return usIdx;
}


/******************************************************************************
 *
 *    Function Name:    Read()
 *
 *    Description:
 *        Reads one bytes from the port and places it into the buffer.
 *
 *    Input Parameters:
 *        None
 *
 *    Output Parameters:
 *        pucData - a pointer to a byte location where the read byte is placed
 *        
 *    Return Value:
 *        1 if the byte read successfully; -1 if error occurred
 *
 *****************************************************************************/
S32 CCommPort::Read (U8 *pucData)
{
    return Read (pucData, 1);
}


/******************************************************************************
 *
 *    Function Name:    SetPort()
 *
 *    Description:
 *        Sets the port number for the port.
 *
 *    Input Parameters:
 *        ucPortNum - port number
 *        
 *    Output Parameters:
 *        None
 *        
 *    Return Value:
 *        true if no error; false otherwise
 *
 *****************************************************************************/
bool CCommPort::SetPort (U8 ucPortNum)
{
    if (ucPortNum > 0) 
    {
        m_ucPortNum = ucPortNum;
        return true;
    }

    return false;
}


/******************************************************************************
 *
 *    Function Name:    SetBaud()
 *
 *    Description:
 *        Sets the baud rate for the port.
 *
 *    Input Parameters:
 *        ulBaud - transmission baud rate
 *        
 *    Output Parameters:
 *        None
 *        
 *    Return Value:
 *        true if no error; false otherwise
 *
 *****************************************************************************/
bool CCommPort::SetBaud (U32 ulBaud)
{
    if ((ulBaud == COM_BAUD_1200) || 
        (ulBaud == COM_BAUD_2400) || 
        (ulBaud == COM_BAUD_4800) || 
        (ulBaud == COM_BAUD_9600) || 
        (ulBaud == COM_BAUD_14k4) || 
        (ulBaud == COM_BAUD_19k2) || 
        (ulBaud == COM_BAUD_28k8) || 
        (ulBaud == COM_BAUD_38k4) || 
        (ulBaud == COM_BAUD_57k6) || 
        (ulBaud == COM_BAUD_115k)) 
    {
        m_ulBaud = ulBaud;
        return true;
    }

    return false;
}


/******************************************************************************
 *
 *    Function Name:    SetStopBits()
 *
 *    Description:
 *        Sets the number of stop bits per byte for the port.
 *
 *    Input Parameters:
 *        ucStopBits - number of stop bits per byte
 *        
 *    Output Parameters:
 *        None
 *        
 *    Return Value:
 *        true if no error; false otherwise
 *
 *****************************************************************************/
bool CCommPort::SetStopBits (U8 ucStopBits)
{
    if ((ucStopBits == COM_STOP_BITS_1) || 
        (ucStopBits == COM_STOP_BITS_2)) 
    {
        m_ucStopBits = ucStopBits;
        return true;
    }

    return false;
}


/******************************************************************************
 *
 *    Function Name:    SetDataBits()
 *
 *    Description:
 *        Sets the number of data bits per byte for the port
 *
 *    Input Parameters:
 *        ucDataBits - number of bits per byte
 *        
 *    Output Parameters:
 *        None
 *        
 *    Return Value:
 *        true if no error; false otherwise
 *
 *****************************************************************************/
bool CCommPort::SetDataBits (U8 ucDataBits)
{
    if ((ucDataBits >= COM_DATA_BITS_4) && 
        (ucDataBits <= COM_DATA_BITS_8))
    {
        m_ucDataBits = ucDataBits;
        return true;
    }

    return false;
}


/******************************************************************************
 *
 *    Function Name:    SetParity()
 *
 *    Description:
 *        Sets the port parity
 *
 *    Input Parameters:
 *        ucParity - port parity
 *        
 *    Output Parameters:
 *        None
 *        
 *    Return Value:
 *        true if no error; false otherwise
 *
 *****************************************************************************/
bool CCommPort::SetParity (U8 ucParity)
{
    if ((ucParity == COM_PARITY_NONE) || 
        (ucParity == COM_PARITY_ODD)  || 
        (ucParity == COM_PARITY_EVEN)) 
    {
        m_ucParity = ucParity;
        return true;
    }

    return false;
}
